package cz.eman.jsonrpc.client;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.ConnectException;

import org.apache.log4j.Logger;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;

import cz.eman.jsonrpc.shared.JsonTransformer;
import cz.eman.jsonrpc.shared.bo.JsonRpcErrorResponse;
import cz.eman.jsonrpc.shared.bo.JsonRpcNormalResponse;
import cz.eman.jsonrpc.shared.bo.JsonRpcRequest;
import cz.eman.jsonrpc.shared.exception.RpcErrorException;
import cz.eman.jsonrpc.shared.exception.RpcException;

public class AbstractClientProxy<T> {

	public static final String CONN_CLOSED = "Connection closed";

	protected static Logger log;

	protected Class<T> clazz;

	protected ClientProvider clientProvider;

	protected boolean strictCheck = true;

	public AbstractClientProxy(final Class<T> clazz, final ClientProvider clientProvider) {
		this(clazz, clientProvider, true);
	}

	public AbstractClientProxy(final Class<T> clazz, final ClientProvider clientProvider, final boolean strictCheck) {
		log = Logger.getLogger(this.getClass());
		this.clazz = clazz;
		this.clientProvider = clientProvider;
		this.strictCheck = strictCheck;
	}

	public void close() throws IOException {
		clientProvider.close();
	}

	protected Method getMethodByName(final String name) {
		for (Method m : clazz.getMethods()) {
			if (m.getName().equals(name)) {
				return m;
			}
		}
		throw new RuntimeException("Unknown method: " + name + " Class: " + clazz);
	}

	protected void checkParameters(final Class<?> clases[], final Object[] params) {
		if (clases.length != params.length) {
			throw new RuntimeException("Parameters length don't match!");
		}
		for (int i = 0; i < params.length; i++) {
			compareClasses(params[i].getClass(), clases[i], i);
		}
	}

	protected void compareClasses(final Class<?> cl1, final Class<?> cl2, final int i) {
		if (cl1 != cl2) {
			if (cl1.isPrimitive() != cl2.isPrimitive()) {

				System.out.println("Warning, possible mismatch between primitive and non-primitive type: " + i + " (" + cl1 + ", " + cl2 + ")");
			} else {
				throw new RuntimeException("Parameter mismatch at: " + i + " (" + cl1 + ", " + cl2 + ")");
			}
		}
	}

	public Object callMethod(final String method, final Object... params) throws JsonParseException, JsonMappingException, IOException {
		return callMethod(method, strictCheck, params);
	}

	public Object callMethod(final String method, final boolean strictCheck_, final Object... params) throws JsonParseException, JsonMappingException, IOException {
		Method m = getMethodByName(method);
		if (strictCheck_) {
			checkParameters(m.getParameterTypes(), params);
		}
		JsonRpcRequest request = new JsonRpcRequest();
		request.setMethod(method);
		request.setId(String.valueOf(System.currentTimeMillis()));
		// TODO: mapa misto pole
		request.setParamsArray(params);
		String content = JsonTransformer.toJson(request);
		String resultText = clientProvider.sendContent(content);
		if (resultText == null) {
			throw new ConnectException(CONN_CLOSED);
		}
		try {
			JsonRpcNormalResponse response = (JsonRpcNormalResponse) JsonTransformer.toObject(resultText, JsonRpcNormalResponse.class);
			if (response.getResult() == null) {
				return null;
			}
			return JsonTransformer.toObject(response.getResult().toString(), m.getReturnType());
		} catch (Exception e) {
			JsonRpcErrorResponse response;
			try {
				response = (JsonRpcErrorResponse) JsonTransformer.toObject(resultText, JsonRpcErrorResponse.class);
				throw new RpcErrorException(response.getError());
			} catch (RpcErrorException e1) {
				throw e1;
			} catch (Exception e2) {
				log.error(e2, e2);
				throw new RpcException(e2);
			}
		}
	}

	public String callNativeMethod(final String method, final Object... params) throws JsonParseException, JsonMappingException, IOException {
		JsonRpcRequest request = new JsonRpcRequest();
		request.setMethod(method);
		request.setId(String.valueOf(System.currentTimeMillis()));
		request.setParamsArray(params);
		String content = JsonTransformer.toJson(request);
		String resultText = clientProvider.sendContent(content);
		if (resultText == null) {
			throw new ConnectException(CONN_CLOSED);
		}
		try {
			JsonRpcNormalResponse response = (JsonRpcNormalResponse) JsonTransformer.toObject(resultText, JsonRpcNormalResponse.class);
			if (response.getResult() == null) {
				return null;
			}
			return response.getResult().toString();
		} catch (Exception e) {
			JsonRpcErrorResponse response;
			response = (JsonRpcErrorResponse) JsonTransformer.toObject(resultText, JsonRpcErrorResponse.class);
			throw new RpcErrorException(response.getError());

		}
	}

}
